
library(shiny)
library(cowplot)
library(ggplot2)
library(dplyr)
library(rlist)
library(signal)

# Define UI for app 
ui <- fluidPage(
  
  # App title ----
  titlePanel("Smoothed time series analysis using Savitsky'ego-Golay filter "),
  
  # Sidebar layout with input and output definitions ----
  sidebarLayout(
    
    # Sidebar panel for inputs ----
    sidebarPanel(
      
      # Input: Slider for the number of bins ----
      sliderInput(inputId = "window_size",
                  label = "Windos size",
                  min = 3,
                  max = 51,
                  step=2,
                  value = 5),
      
      sliderInput(inputId = "degree",
                  label = "Degree",
                  min = 1,
                  max = 10,
                  step=1,
                  value = 2)
      
    ),
    
    # Main panel for displaying outputs ----
    mainPanel(
      
      plotOutput(outputId = "distPlot")
      
    )
  )
)

# Define server logic 
server <- function(input, output) {
  observe({
    updateSliderInput(inputId="degree",max=input$window_size-1)
  })

  # This expression that generates a histogram is wrapped in a call
  # to renderPlot to indicate that:
  #
  # 1. It is "reactive" and therefore should be automatically
  #    re-executed when inputs (input$bins) change
  # 2. Its output type is a plot
  output$distPlot <- renderPlot({
    
    
    df=data.frame(year=as.numeric(time(Nile)),flow=as.numeric(Nile),smooth=sgolayfilt(Nile,n=input$window_size,p=input$degree,m=0,ts=1))
    
    xborder1=df[(input$window_size-1)/2,'year']
    xborder2=df[length(Nile)-(input$window_size-1)/2,'year']
    
 
    
    
    ggplot()+
      annotate("rect",xmin=-Inf,xmax=time(Nile)[(input$window_size-1)/2],ymin=-Inf,ymax=Inf,fill="red",alpha=0.1)+
      annotate("rect",xmin=xborder2,xmax=Inf,ymin=-Inf,ymax=Inf,fill="red",alpha=0.1)+
      xlab("Year")+ylab("Water flow in Nile river (10^8 m^3)")+
      geom_line(data=df, aes(x=year,y=flow),color="Blue",size=1.1)+geom_line(data=df, aes(x=year,y=smooth),color="green",size=2)
      
      
    
    
    
  })
  
}

# Create Shiny app ----
shinyApp(ui = ui, server = server)